<?php
/**
 * Конструктор HTML для загрузки библиотек
 *
 * Загрузка JS Библиотек
 * Загрузка CSS Библиотек
 *
 * @param array $delayed
 *
 * @author veselka.ua
 * @version 0.6
 *
 * @package veselka_ua/themes
 */

class WpLibconstruct {


	use PluginsPath;

	/**
	 * Список соответствующих библиотек в виде массива опций.
	 * @type array
	 */
	private $lib_list = [];
	/**
	 * Флаг отложенной загрузки.
	 * @type string
	 */
	private $lazy_loading;
	/**
	 * Параметры отложенной загрузки.
	 * @type array
	 */
	private $delayed;
	/**
	 * HTML обертки для вызова библиотек.
	 * @type array
	 */
	private $wrapper;
	/**
	 * Дополнения к библиотекам.
	 * @type array
	 */
	private $add;
	/**
	 * Связанные библиотеки.
	 * @type array
	 */
	private $associated;
	/**
	 * Шаблоны исключения.
	 * @type array
	 */
	private $exсlude = [];
	/**
	 * Шаблоны исключения.
	 * @type array
	 */
	private $inсlude = [];


	public function __construct() {

		// Проверка наличия плагина
		// Плагины могут быть дочерними (частными случаями)
		// Имена констрант с путями могут совпадать для разных имен плагинов
		$this->check_plugins();

		// Инициализация
		$this->load();

		/**
		 * Передача текущего объекта в фильтр по ссылке для объктов использующих общие методы в плагинах
		 */
		foreach( $this->plugins_tax as $recipient => $plugin ) {
			if( has_action( $recipient.'_lib_exchange' ) ) {
				do_action_ref_array( $recipient.'_lib_exchange', [&$this] );
			}
		}

		/**
		 * Хук для методов в шаблоне
		 * Передача текущего объекта в фильтр по ссылке для объктов использующих общие методы в шаблоне
		 */
		do_action_ref_array( 'wp_lib_construct_exchange', [&$this] );


		// Вывод элементов в хедере
		add_action( 'header_css', [$this, 'get_head_elements'], 50 );
		// Вывод элементов css в футере
		add_action( 'footer_css', [$this, 'get_footer_css'], 1 );
		// Вывод элементов js в футере
		add_action( 'footer_js', [$this, 'get_footer_js'], 1 );
	}


//////////////////// Публичные методы ////////////////////


	/**
	 * Обертка для крючка в хедере (hook) вордпресс.
	 *
	 * @param void
	 * @return void
	 */
	final public function get_head_elements() {
		echo $this->lib_construct('css');
	}


	/**
	 * Обертка для вывода элементов css в футере.
	 *
	 * @param void
	 * @return string
	 */
	final public function get_footer_css() {
		echo $this->lib_construct('css');
	}


	/**
	 * Обертка для вывода элементов js в футере.
	 *
	 * @param void
	 * @return string
	 */
	final public function get_footer_js() {
		echo $this->lib_construct('js');
	}

	/**
	 * Обертка для записи новых элементов в массив библиотек.
	 *
	 * @param array|string $lib_list
	 * @param string $lib
	 * @return void
	 */
	final public function set_lib_list( $lib_list = false, $lib = false, $exclude = false, $include = false  ) {

		if( !$lib || !$lib_list ) {
			return false;
		}

		// Передан не массив
		if( !is_array($lib_list) ) {
			$lib_list = [$lib_list => true];
		}
		// Регистрация библиотек
		foreach( $lib_list as $key => $value ) {
			if( $value === 'uploaded' ) {
				continue;
			}
			$this->lib_list[$lib][$key] = $value;

			// Исключения
			if($exclude) {
				$this->exclude[$key] = $exclude;
			} elseif( isset($this->exclude[$key]) ) {
				unset($this->exclude[$key]);
			}
			if($include) {
				$this->include[$key] = $include;
			} elseif( isset($this->include[$key]) ) {
				unset($this->include[$key]);
			}
		}
	}


	/**
	 * Обертка для записи новых элементов в массив библиотек.
	 *
	 * @param array|string $lib_list
	 * @param array $exclude
	 * @param array $include
	 * @return void
	 */
	final public function set_new_lib( $lib_name = false, $exclude = false, $include = false ) {

		if( !$lib_name ) {
			return false;
		}

		// CSS
		$this->set_lib_list( $lib_name, 'css', $exclude, $include );
		// JS
		$this->set_lib_list( $lib_name, 'js', $exclude, $include );
	}


	/**
	 * Обертка для дополнения.
	 *
	 * @param array|string $lib_list
	 * @param string $add
	 * @return void
	 */
	final public function set_addon( $lib_name = false, $add = false ) {

		if( !$lib_name ) {
			return false;
		}
		if( !$add || $add == '' ) {
			return false;
		}
		$this->add[$lib_name] .= $add;
	}



//////////////////// Приватные и защищенные методы ////////////////////


	/**
	 * Развертывание.
	 *
	 * @param void
	 * @return void
	 */
	private function load() {

		// Конфигурация из файла
		$this->get_config();

		// Отложенная загрузка
		$this->lazy_loading = false;

		// Части HTML кода для подключения библиотек
		$this->wrapper['js']['begin'] = '<script src="';
		$this->wrapper['js']['end'] = '"></script>';
		$this->wrapper['css']['begin'] = '<link rel="stylesheet" type="text/css" href="';
		$this->wrapper['css']['end'] = '" media="screen" />';

		// Связанные библиотеки
		$this->associated['highslide'] = [
			'highslide.mobile' => true,
			'highslide-with-gallery' => true,
		];
	}


	/**
	 * Библиотеки заранее прописанные в конфиге.
	 *
	 * @param void
	 * @return void
	 */
	private function get_config() {
		/**
		 * Стандартный путь к параметрам.
		 */
		require_once TEMPLATEPATH . '/conf/libs.php';

		// Скрипты
		if( isset($js) ) {
			// Запись списка библиотек в объект конструктор разметки библиотек.
			$this->set_lib_list( $js, 'js' );
		}
		// Стили
		if( isset($css) ) {
			// Запись списка библиотек в объект конструктор разметки библиотек.
			$this->set_lib_list( $css, 'css' );
		}
		// Отложенная загрузка библиотек
		if( isset($delayed) ) {
			$this->delayed = $delayed;
		}

	}


	/**
	 * Конструктор кода подключения библиотек.
	 *
	 * В качестве аргумента метода передется тип виблиотек (js или css).
	 * @param string $lib
	 * @return void
	 */
	private function lib_construct($lib) {
		// Исключения
		$this->lib_exceptions();
		// Аудит списка библиотек
		$this->lib_inspector($lib);
		// Исключения
		$this->lib_associated($lib);
		// Формирование путей к библиотекам
		$this->lib_path_construct($lib);
		// Формирование путей к плагинам
		$this->lib_plugins_path_construct();
		// Конструктор кода
		return $this->lib_html($lib);

	}


	/**
	 * Исключения.
	 * 
	 * Библиотеки загружаемые с условиями.
	 *
	 * @param void
	 * @return void
	 */
	private function lib_exceptions() {

		/**
		 * Если страница контактов загружам карты.
		 */
		if( is_page_template('templates/contacts.php') || is_page_template('templates/contacts-multiple.php') ) {
			// Библиотеки
			$this->lib_list['js']['google_maps'] = 'https://maps.google.com/maps/api/js';
			$this->lib_list['js']['map.multiple'] = true;
			$this->lib_list['css']['map.multiple'] = true;
		}

		/**
		 * Если страница лендинга загружам карты.
		 */
		if( is_page_template('landing.php') ) {
			// Библиотеки
			$this->lib_list['js']['google_maps'] = 'https://maps.google.com/maps/api/js';
			$this->lib_list['js']['map.multiple'] = true;
			$this->lib_list['css']['map.multiple'] = true;
		} else {
			unset($this->lib_list['js']['landing']);
			unset($this->lib_list['css']['landing']);
			unset($this->lib_list['js']['counter']);
			unset($this->lib_list['css']['counter']);
		}

		/**
		 * Если маркетинговые инструменты включены - нужно загрузить js механизмы и стили.
		 */
		if( (bool)get_option('common_modal_forms') ) {
			$this->set_lib_list( ['modal_form' => true], 'js' );
			$this->set_lib_list( ['modal_form' => true], 'css' );
		}

		/**
		 * Комментарии если включены.
		 */
		if( is_singular() && comments_open() && get_option('thread_comments') == 1 ) {
			$lib = get_home_url( null, '/wp-includes/js/comment-reply.min.js' );
			$this->set_lib_list( ['comments' => $lib], 'js' );
		}

		/**
		 * Ключи Гугл карт. Если нет ключа, библиотеку не загружать.
		 */
		if( isset($this->lib_list['js']['google_maps']) && $this->lib_list['js']['google_maps'] != '' ) {
			// Поиск ключа
			$key = get_option('map_api_key');
			if( $key && $key != '' ) {
				$key = '?key=' . $key;
				$this->lib_list['js']['google_maps'] .= $key;
			} else {
				unset($this->lib_list['js']['google_maps']);
				unset($this->lib_list['js']['map.multiple']);
				unset($this->lib_list['css']['map.multiple']);
			}
		}

	}


	/**
	 * Инспектор списка библиотек.
	 *
	 * Регулирует загрузку библиотек, из списка в опциях, необходимых для определенных страниц.
	 * Удаляет из списка библиотеки для конкретного типа страницы.
	 * 
	 * Проверка основана на определении типа страницы, поэтому должна выполнятся внутри цикла вордпресс.
	 * На этапе формирования списка библиотек это сделать невозможно.
	 *
	 * @param string $lib
	 * @return void
	 */
	private function lib_inspector($lib) {
		
		// Если текущая страница использует уникальный шаблон - пытаемся загрузить одноименнsый js/css файл
		$tmp_name = get_page_template_slug();
		if( $tmp_name ) {
			$lib_name = explode('.', $tmp_name);
			if( !isset($this->lib_list[$lib][$lib_name[0]]) ) {
				$this->lib_list[$lib][$lib_name[0]] = true;
			}
		}

		// Исключения, заданные при подключении
		if( $tmp_name && !empty($this->exclude) ) {
			foreach( $this->exclude as $key => $exclude ) {
				if( in_array($tmp_name, $exclude) ) {
					unset($this->lib_list[$lib][$key]);
				}
			}
		}
		if( !empty($this->include) ) {
			foreach( $this->include as $key => $include ) {
				if( !$tmp_name || !in_array($tmp_name, $include) ) {
					unset($this->lib_list[$lib][$key]);
				}
			}
		}
	}


	/**
	 * Ассоциарованные библиотеки.
	 *
	 * В качестве аргумента метода передется тип виблиотек (js или css).
	 * @param string $lib
	 * @return void
	 */
	private function lib_associated($lib) {
		// Перебор
		foreach( $this->lib_list[$lib] as $key => $name ) {
			// Поиск дополнений
			if( isset($this->associated[$key]) ) {
				$this->lib_list[$lib] = array_merge( $this->lib_list[$lib], $this->associated[$key] );
			}
		}
	}


	/**
	 * Формирование путей к библиотекам.
	 *
	 * В качестве аргумента метода передется тип виблиотек (js или css).
	 * @param string $lib
	 * @return void
	 */
	private function lib_path_construct($lib) {

		// Формирование путей к библиотекам
		foreach( $this->lib_list[$lib] as $key => $name ) {

			// Уже загружено?
			if(!$name) {
				unset($this->lib_list[$lib][$key]);
				continue;
			}
			// Убирает из массива элеметы равные false
			if( $name === 'uploaded' || ( !in_array($key,$this->delayed) && !$this->lazy_loading ) ) {
				continue;
			}
			// Проверяет наличие указанного CDN
			$cdn = explode("//", $name);
			if( $cdn[0] == 'http:' || $cdn[0] == 'https:' || $cdn[0] == '' ) {
				continue;
			}
			// Проверяет существуют ли необходимые файлы
			if($name === true) {
				if( is_file( TEMPLATEPATH . '/assets/' . $lib . '/' . $key . '.min.' . $lib ) ) {
					$this->lib_list[$lib][$key] = TEMPLATE_URL . '/assets/' . $lib . '/' . $key . '.min.' . $lib;
				} elseif( is_file( TEMPLATEPATH . '/assets/' . $lib . '/' . $key . '.' . $lib ) ) {
					$this->lib_list[$lib][$key] = TEMPLATE_URL . '/assets/' . $lib . '/' . $key . '.' . $lib;
				} else {
					unset($this->lib_list[$lib][$key]);
				}
			} else {
				$file = explode(".", $name);
				if( array_pop($file) == $lib && is_file( TEMPLATEPATH . '/assets/' . $lib . '/' . $name ) ) {
					$this->lib_list[$lib][$key] = TEMPLATE_URL . '/assets/' . $lib . '/' . $name;
				} else {
					unset($this->lib_list[$lib][$key]);
				} 
			}
		}//foreach
	}


	/**
	 * Конструктор путей.
	 *
	 * Список плагинов и путей к ним формируется в общем объекте PluginsPath
	 * В данном методе происходит только оборачивание в html
	 *
	 * @param void
	 * @return void
	 */
	private function lib_plugins_path_construct() {

		// Дополнения, путь каталогу шаблона
		$this->add['js'] = "\t" . '<script>var template_dir = "' . TEMPLATE_URL . '"</script>' . PHP_EOL;

		$p_arr = get_option( 'active_plugins');
		if( !empty($this->plugins_url) ) {
			foreach( $this->plugins_url as $plugin => $url ) {
				// Путь к каталогу плагина
				$this->add['js'] .= "\t" . '<script>var ' . $plugin . '_dir = "' . $url . '"</script>' . PHP_EOL;
	
			}
		}
		$this->add['css'] = '';
		$this->add['lazy_loading'] = 'type="text/css" property="stylesheet"';
	}



	/**
	 * Конструктор кода.
	 *
	 * В качестве аргумента метода передется тип виблиотек (js или css).
	 * @param string $lib
	 * @return void
	 */
	private function lib_html($lib) {

		$result_set = '';
		// HTML обертка для подключения библиотек (вынесено в дополнительный проход циклом по опциям для читабельности нетривиальной части кода выше)
		foreach( $this->lib_list[$lib] as $key => $name ) {

			// Убирает из массива элеметы равные false
			if( $name === 'uploaded' || ( !in_array($key,$this->delayed) && !$this->lazy_loading ) ) {
				continue;
			}

			$link = $this->wrapper[$lib]['begin'] . $name . $this->wrapper[$lib]['end'];
			// Плюшка для http://validator.w3.org/ в случае когда css не в хедере
			if( $lib === 'css' && $this->lazy_loading ) {
				$link = str_replace( 'type="text/css"', $this->add['lazy_loading'], $link );
			}
			// HTML комментарии - обертка библиотек подключаемых только для IE
			if( $key == 'html5shiv' || $key == 'respond' ) {
				$link = '<!--[if lt IE 9]>' . $link . '<![endif]-->';
			} elseif( $key == 'highslide-ie6' ) {
				$link = '<!--[if lt IE 7]>' . $link . '<![endif]-->';
			}
			$result_set .= "\t" . $link . PHP_EOL;
			$this->lib_list[$lib][$key] = 'uploaded';
		}//foreach

		// Добавление HTML комментария - заголовка блока бибилиотек
		if( $lib === 'css' ) {
			$this->lazy_loading = true;
		}
		if( $result_set ) {
			return PHP_EOL . "\t" . '<!-- ' . strtoupper($lib) . ' -->' . PHP_EOL . $this->add[$lib] . $result_set;
		} else {
			return false;
		}
	}

} //end WpLibconstruct
?>