<?php
/**
 * Загрузка шаблонов.
 * 
 * Общие механизмы для обработки файлов шаблонов.
 * Работает только в паре с примесью FileHandling.
 *  
 * @author veselka.ua
 * @version 0.1
 *
 * @package veselka_ua/themes
 */

trait TemplatesLoad {

	/**
	 * Результат работы.
	 * HTML разметка шаблонов.
	 *
	 * @type array
	 */
	private $templates;

	/**
	 * Получение разметки шаблона.
	 *
	 * Принимает ключ массива с путями или путь целиком.
	 * Возвращает шаблон или false в случае его отсутствия.
	 *
	 * @param string $file_name
	 * @param string $type
	 * @return string|bool
	 */
	final protected function get_template( $file_name, $type = false ) {
		$this->check_template( $file_name, $type );
		return $this->templates[$file_name];
	}

	/**
	 * Проверка наличия разметки.
	 *
	 * Принимает ключ массива с путями или путь целиком.
	 * Возвращает true или false в случае отсутствия шаблона.
	 *
	 * @param string $file_name
	 * @param string $type
	 * @return bool
	 */
	final protected function check_template( $file_name, $type = false ) {
		if( !isset($this->templates[$file_name]) ) {
			$file_path = $this->check_file_path( $this->options['files'][$file_name], $type, true );
			if( !$file_path ) {
				$this->templates[$file_name] = false;
				return false;
			}
			$this->templates[$file_name] = file_get_contents( $file_path );
		}

		return true;
	}


	/**
	 * Персонализация HTML разметки.
	 * Замена маркеров в шаблоне на значения из массива аргументов.
	 * Включает в себя проверку наличия шаблона.
	 * 
	 * @param array $replacement - массив замен.
	 * @param string $template - html разметка включающая маркеры для замены.
	 * @param string|bool $type - тип шаблона
	 * @return string|bool
	 */
	final protected function template_replacement($replacement, $template, $type = false) {
		if( !isset($replacement) || !isset($template) || $template == '' || !is_array($replacement) ) {
			return false;
		}
		if( !$this->check_template( $template, $type ) ) {
			return false;
		}
		return $this->markers_replacement( $this->templates[$template], $replacement );
	}


	/**
	 * Замена маркеров.
	 * 
	 * @param string|bool $incubator - html разметка включающая маркеры для замены.
	 * @param array $replacement - массив замен.
	 * @return string|bool
	 */
	final protected function markers_replacement($incubator, $replacement) {
		
		if( !$incubator ) {
			return false;
		}
		// Замена маркеров значениями
		foreach( $replacement as $key => $value ) {
			// Маркер
			$marker = '[%' . strtoupper($key) . '%]';
			// Замена текущего маркера
			$incubator = str_replace($marker, $value, $incubator);
		}

		return $incubator;
	}

} // end TemplatesLoad
?>