<?php
/**
 * Работа с файлами.
 * 
 * Общие механизмы для обработки загружаемых файлов.
 *  
 * @author veselka.ua
 * @version 0.2
 *
 * @package veselka_ua/themes
 */

trait FileHandling {

	/**
	 * Пути к файлам.
	 * @type array
	 */
	private $path = [];


//////////////////// Защищенные методы ////////////////////

	/**
	 * Проверка наличия каталога.
	 *
	 * Принимает ключ массива с путями или путь целиком.
	 * Возвращает путь в случае успеха и false в противном случае.
	 *
	 * @param string $folder_path
	 * @return mixed string|bool
	 */
	final protected function check_folder( $folder_path = false ) {

		// Если не указан путь
		if( !$folder_path ) {
			return false;
		}

		if( strpos($folder_path, '/') !== false ) {
			// Пользовательская директория существует?
			if( !is_dir( $folder_path ) ) {
				return false;
			}
		} else {
			$folder_path = $this->construct_folders_path( $folder_path );
			// Указанныый путь не существует
			if( !$folder_path ) {
				return false;
			}
		}

		// Читабельно
		if( !is_readable( $folder_path ) ) {
			return false;
		}

		// Есть файлы
		$handle = opendir( $folder_path );
		while( ($entry = readdir($handle)) !== false ) {
			if( $entry != "." && $entry != ".." ) {
				return $folder_path;
			}
		}
		// Файлов не оказалось
		return false;
	}


	/**
	 * Проверка файлов.
	 *
	 * В качестве аргументов принимает имя файла и ключ массива с путями. Возможно задать пользовательский путь вместо ключа.
	 * @param string $file_name
	 * @param string $type
	 * @param bool $path - вернуть путь или только проверить
	 * @return bool
	 */
	final protected function check_file_path( $file_name, $type = false, $path = false ) {

		// Если не указан путь - искать файл в корне
		if( !$type ) {
			$type = '/';
		}
		$sub_path = $this->check_folder( $type );

		// Проверка переменных
		if( !$sub_path ) {
			return false;
		}
		if( !$file_name ) {
			return false;
		}

		// Проверка наличия файла
		$file_path = $sub_path . $file_name;
		if( file_exists($file_path) ) {
			if( $path ) {
				return $file_path;
			} else {
				return true;
			}
		} else {
			return false;
		}
	}


//////////////////// Приватные методы ////////////////////

	/**
	 * Проверка наличия системных каталогов.
	 * Или поиск каталога если еще не установлен.
	 *
	 * @param string $folder - маркер массива путей.
	 * @return mixed string|bool
	 */
	private function construct_folders_path( $folder ) {

		// Проверка текущего каталога осуществлялась
		if( isset($this->path[$folder]) && $this->path[$folder] !== false ) {
			return $this->path[$folder];
		}
		if( isset($this->path[$folder]) && $this->path[$folder] === false ) {
			return false;
		}

		// CSV
		if( $folder == 'csv' ) {
			// Возможные пути
			$sub_path[0] = $_SERVER["DOCUMENT_ROOT"];
			$sub_path[1] = TEMPLATEPATH;
			foreach($sub_path as $path ) {
				$path = $path . '/' . $folder . '/';
				if( is_dir($path) ) {
					$this->path['csv'] = $path;
					return $path;
				}
			}
			// Ничего нет
			return false;
		}

		// Стандартные каталоги
		$folder_path = TEMPLATEPATH . '/' . $folder . '/';
		if( is_dir($folder_path) ) {
			$this->path[$folder] = $folder_path;
			return $folder_path;
		}

		// Запчасти
		$folder_path = TEMPLATEPATH . '/view/' . $folder . '/';
		if( is_dir($folder_path) ) {
			$this->path[$folder] = $folder_path;
			return $folder_path;
		}

		// Виджеты
		$folder_path = TEMPLATEPATH . '/view/widgets/' . $folder . '/';
		if( is_dir($folder_path) ) {
			$this->path[$folder] = $folder_path;
			return $folder_path;
		}

		// Варианты закончились
		return false;
	}

} // end FileHandling
?>