<?php
/**
 * Генерация XML карты сайта.
 *
 * Генерирует карту сайта на основе имеющихся страниц, постов, таксономий и пользовательских типов записей.
 * Оповещает поисковые системы о изменениях в карте сайта.
 * Возможна генерация карты сайта для фейковых страниц, для этого нужно передать соответствующие параметры в объект.
 *
 * @author veselka.ua
 * @version 0.3
 *
 * @package veselka_ua/themes
 */

class SeoSitemap {

	/**
	 * Массив параметров
	 * @type array
	 */
	private $options = [];
	/**
	 * Массив генерируемых страниц
	 * @type array
	 */
	private $fake_slugs = [];
	/**
	 * Массив генерируемых ссылок
	 * @type array
	 */
	private $fake_parent_url = [];


	public function __construct($fake_slugs = false, $fake_parent_url = false) {

		// Данные несуществующих страниц
		if( $fake_slugs && !empty($fake_slugs) && is_array($fake_slugs) ) {
			$this->fake_slugs = $fake_slugs;
		}
		if( $fake_parent_url && !empty($fake_parent_url) && is_array($fake_parent_url) ) {
			$this->fake_parent_url = $fake_parent_url;
		}

		/**
		 * Хуки вордпресс
		 */
		add_action('publish_post', [$this, 'load_xml_sitemap']);
		add_action('publish_page', [$this, 'load_xml_sitemap']);
		// Режим отладки, запись карты при сохранении изменений
		if( (bool)get_option('xmlsitemap_debug') ) {
			add_action('save_post', [$this, 'load_xml_sitemap']);
			add_action('save_page', [$this, 'load_xml_sitemap']);
		}
	}


	/**
	 * Обертка.
	 *
	 * @param void
	 * @return void
	 */
	final public function load_xml_sitemap() {
		// Параметры
		$this->load_options();
		// Генерация карты сайта
		$this->xml_formalization();
	}


	/**
	 * Запись параметров.
	 *
	 * @param void
	 * @return void
	 */
	private function load_options() {
		/**
		 * Запись параметров, возможно перенести к основным параметрам в админку.
		 */
		$this->options['sitemap_file_name'] = 'sitemap.xml'; // Имя файла для карты сайта (обязательно установить права на запись)

		// Что включть в карту сайта
		$this->options['map_home'] = true; // Включить главную. Значения: true|false
		$this->options['map_post'] = true; // Включить записи. Значения: true|false
		$this->options['map_products'] = true; // Включить товары. Значения: true|false
		$this->options['map_estate'] = false; // Включить квартиры. Значения: true|false
		$this->options['map_page'] = true; // Включить страницы. Значения: true|false
		$this->options['map_category'] = true; // Включить категории. Значения: true|false
		$this->options['map_catalog'] = true; // Включить категории товаров. Значения: true|false
		$this->options['map_realty'] = true; // Включить районы. Значения: true|false
		$this->options['map_fake'] = false; // Включить генерацию. Значения: true|false

		// Приоритеты
		$this->options['priority_home'] = 1; // Приоритет главной. Значения: 1 - 0.1
		$this->options['priority_post'] = 0.7; // Приоритет записей. Значения: 1 - 0.1
		$this->options['priority_products'] = 0.7; // Приоритет товаров. Значения: 1 - 0.1
		$this->options['priority_estate'] = 0.7; // Приоритет апартаментов. Значения: 1 - 0.1
		$this->options['priority_page'] = 0.5; // Приоритет страниц. Значения: 1 - 0.1
		$this->options['priority_category'] = 0.4; // Приоритет категорий. Значения: 1 - 0.1
		$this->options['priority_catalog'] = 0.4; // Приоритет категорий твоаров. Значения: 1 - 0.1
		$this->options['priority_realty'] = 0.5; // Приоритет категорий. Значения: 1 - 0.1
		$this->options['priority_fake'] = 0.3; // Приоритет генерируемого контента. Значения: 1 - 0.1

		// Исключить по id из карты сайта. Значения: true|false|string
		$this->options['exclude_post'] = false; // Список постов
		$this->options['exclude_page'] = false; // Список страниц
		$this->options['exclude_estate'] = false; // Список квартир
		$this->options['exclude_products'] = false; // Список товаров
		$this->options['exclude_category'] = false; // Список категорий
		$this->options['exclude_catalog'] = false; // Список категорий товаров
		$this->options['exclude_realty'] = false; // Список районов
	}


	/**
	 * Формирование XML разметки карты сайта.
	 *
	 * @param void
	 * @return void
	 */
	private function xml_formalization() {
		// Заголовок документа
		$sitemap = "<?xml version='1.0' encoding='UTF-8'?>" . PHP_EOL;
		// Cтили отображения XML карты, для удобного просмотра человеком
		if( (bool)get_option('xmlsitemap_stylesheet') ) {
			$sitemap .= "<?xml-stylesheet type='text/xsl' href='" . TEMPLATE_URL . "/inc/stylesheet-xml-sitemap.xsl'?>" . PHP_EOL;
		}
		$sitemap .= "<!--generated-on='" . date("Y-m-d H:i:s") . "'-->" . PHP_EOL;
		// Открывающий тег узлов карты сайта
		$sitemap .= "<urlset xmlns='http://www.sitemaps.org/schemas/sitemap/0.9'>" . PHP_EOL;

		// Формирование xml разметки для узлов карты сайта
		foreach( $this->options as $key => $value ) {
			$type = explode( '_', $key );
			if( $type[0] == 'map' && $value ) {
				// Формирование имени метода в зависимости от текущего элемента.
				$method_name = $type[1] . '_construct';
				// Вызов соответствующего метода для сбора прописанных метданнх.
				if( method_exists( $this, $method_name ) ) {
					$sitemap .= $this->{$method_name}();
				}
			}
		}

		// Закрывающий тег
		$sitemap .= '</urlset>';
		// Запись карты сайта
		$this->sitemap_filing( $sitemap );
	}


	/**
	 * Главная страница.
	 *
	 * @param void
	 * @return void
	 */
	private function home_construct() {
		$url = get_option('home');
		$last_mod = date('Y-m-d\TH:i:sP');
		$sitemap .= $this->node_formalisation($url, $last_mod, 'home');
		return $sitemap;
	}


	/**
	 * Список постов для карты сайта.
	 *
	 * @param void
	 * @return string
	 */
	private function post_construct() {
		//$sitemap = '';
		// Формирование параметров записей
		return $this->single_construct('post');
	}


	/**
	 * Список страниц для карты сайта.
	 *
	 * @param void
	 * @return string
	 */
	private function page_construct() {
		$this->options['home_id'] = get_option('page_on_front');
		if( $this->options['home_id'] != '0' ) {
			$this->options['exclude_page'][] = $this->options['home_id'];
		}
		// Формирование параметров записей
		return $this->single_construct('page');
	}


	/**
	 * Список квартир для карты сайта.
	 *
	 * @param void
	 * @return string
	 */
	private function estate_construct() {
		// Формирование параметров записей
		return $this->single_construct('estate');
	}


	/**
	 * Список товаров для карты сайта.
	 *
	 * @param void
	 * @return string
	 */
	private function products_construct() {
		// Формирование параметров записей
		return $this->single_construct('products');
	}


	/**
	 * Список единичных записей.
	 * Может быть page, post, products и т.п.
	 *
	 * @param string $type
	 * @return string $sitemap
	 */
	private function single_construct($type) {
		$sitemap = '';
		// Формирование параметров записей
		$exclude = [];
		if( is_array($this->options['exclude_'.$type]) && !empty($this->options['exclude_'.$type]) ) {
			$exclude = $this->options['exclude_'.$type];
		}
		$args = [
			'posts_per_page'	=>	-1,
			'post_type'			=>	$type,
			'orderby'			=>	'modified',
			'order'				=>	'DESC',
			'post__not_in'		=>	$this->options['exclude_'.$type],
			'post_status'		=>	'publish',
		];
		// Запись данных в общий массив
		$sitemap_query = new WP_Query( $args );
		if ( $sitemap_query->have_posts() ) {
			// Цикл Вордпресс
			while ( $sitemap_query->have_posts() ) : $sitemap_query->the_post();
				//if($type = 'page') {
					//$url = get_permalink($sitemap_query->post->ID);
				//} else {
					$url = get_permalink($sitemap_query->post->ID);	
				//}
				$last_mod = date('Y-m-d\TH:i:sP', strtotime($sitemap_query->post->post_modified));
				$sitemap .= $this->node_formalisation($url, $last_mod, $type);
			endwhile;
		} // endif
		wp_reset_postdata();
		return $sitemap;
	}


	/**
	 * Список категорий для карты сайта.
	 *
	 * @param void
	 * @return string
	 */
	private function category_construct() {
		return $this->tax_construct('category','post');
	}


	/**
	 * Список категорий для карты сайта.
	 *
	 * @param void
	 * @return string
	 */
	private function realty_construct() {
		return $this->tax_construct('realty','estate',true);
	}


	/**
	 * Список категорий для карты сайта.
	 *
	 * @param void
	 * @return string
	 */
	private function catalog_construct() {
		return $this->tax_construct('catalog','products',true);
	}


	/**
	 * Список таксономий для карты сайта.
	 *
	 * @param string $tax_name
	 * @return string $sitemap
	 */
	private function tax_construct($tax_name, $post_type, $hide_empty=false) {
		$sitemap = '';
		$last_mod_time = 0;
		if($hide_empty) {
			$taxonomy = get_terms($tax_name, ['hide_empty'=>false,]);
		} else {
			$taxonomy = get_terms($tax_name);
		}

		foreach( $taxonomy as $tax ) {
			$url = get_category_link($tax->term_id);
			// Поиск последнего модифицированного поста
			$posts = get_posts([
				'category'		=>	$tax->term_id,
				'numberposts'	=>	-1,
				'orderby'		=>	'modified',
				'post_type'		=>	$post_type,
				'order'			=>	'DESC',
			]);
			if(!empty($posts)) {
				foreach($posts as $post) {
					setup_postdata($post);
					$mod_time = strtotime($post->post_modified);
					if( $mod_time > $last_mod_time ) {
						$last_mod_time = $mod_time;
					}
				}
				wp_reset_postdata();
			} else {
				$tax_data = get_option("taxonomy_$tax->term_id");
				if($tax_data['cat_modify']) {
					$last_mod_time = strtotime($tax_data['cat_modify']);
				} else {
					$post = get_post($this->options['home_id']);
					$last_mod_time = strtotime($post->post_modified);
				}
			}
			//date_format($date, 'Y-m-d H:i:s');
			// Преобразование даты
			$last_mod_time = date('Y-m-d\TH:i:sP', $last_mod_time);
			// Запись результата
			$sitemap .= $this->node_formalisation($url, $last_mod_time, $tax_name);
		}
		return $sitemap;
	}


	/**
	 * Список генерируемых страниц для карты сайта.
	 *
	 * @param void
	 * @return sring $sitemap
	 */
	private function fake_construct() {
		$sitemap = '';
		if( !$this->fake_slugs ) {
			return $sitemap;
		}
		foreach( $this->fake_slugs as $iata_code => $slug_from ) {
			foreach( $this->fake_slugs as $iata_code_to => $slug_to ) {
				if( $slug != $slug_to ) {
					$url = $this->fake_parent_url . $slug_from . '-to-' . $slug_to . '.html';
					// Генерация номера месяца
					$month = ord( substr( $slug_from, 1, 1) );
					while( $month > 12 ) {
						$month = $month / 3 * 2 - 5;
					}
					$month = round($month);
					// Генерация номера дня
					$days = ord( substr( $slug_from, 2, 1) );
					while( $days > 28 ) {
						$days = $days / 3 * 2  - 16;
					}
					$days = round($days);
					// Генерация часов
					$hours = ord( substr( $slug_to, 1, 1) );
					while( $hours >= 23 ) {
						$hours = $hours / 3 * 2 - 12;
					}
					$hours = round($hours);
					// Генерация минут
					$minutes = ord( substr( $slug_to, 2, 1) );
					while( $minutes >= 59 ) {
						$minutes = $minutes / 3 * 2 - 32;
					}
					$minutes = round($minutes);
					// Генерация секунд
					$seconds = ord( substr( $slug_to, 1, 1) );
					while( $seconds >= 59 ) {
						$seconds = $seconds / 3 * 2 - 32;
					}
					$seconds = round($seconds);
					$fake_date = strtotime('2016-' . $month . '-' . $days . ' ' . $hours . ':' . $minutes . ':' . $seconds . ' UTC');
					$last_mod = date( 'Y-m-d\TH:i:sP', $fake_date );
					$sitemap .= $this->node_formalisation($url, $last_mod, 'fake');
				}
			}
		}
		return $sitemap;
	}


	/**
	 * Формирование XML кода узла.
	 *
	 * @param string $url
	 * @param date $date
	 * @param string $type
	 * @return string $sitemap
	 */
	private function node_formalisation($url, $date, $type) {
		$sitemap = '';
		if( $url && $url != '' ) {
			$sitemap .= "\t" . '<url>' . PHP_EOL;
			$sitemap .= "\t\t" . '<loc>' . $url . '</loc>' . PHP_EOL;
			$sitemap .= "\t\t" . '<lastmod>' . $date . '</lastmod>' . PHP_EOL;
				if( $this->options['priority_' . $type] == 1 ) {
					$changefreq = 'daily';
				} elseif ( $this->options['priority_' . $type] > 0.5 ) {
					$changefreq = 'weekly';
				} elseif ( $this->options['priority_' . $type] <= 0.5 ) {
					$changefreq = 'monthly';
				}
			$sitemap .= "\t\t" . '<changefreq>' . $changefreq . '</changefreq>' . PHP_EOL;
			$sitemap .= "\t\t" . '<priority>' . $this->options['priority_' . $type] . '</priority>' . PHP_EOL;
			$sitemap .= "\t" . '</url>' . PHP_EOL;
		}
		return $sitemap;
	}


	/**
	 * Резервирование карты сайта.
	 *
	 * 1. Запись XML в файл.
	 * 2. Оповещение поисковиков.
	 *
	 * @param string $sitemap
	 * @return void
	 */
	private function sitemap_filing( $sitemap ) {

		// Запись карты сайта в файл
		$file = fopen( str_replace('wp/','',ABSPATH) . $this->options['sitemap_file_name'], 'w' );
		fwrite( $file, $sitemap );
		fclose( $file );

		// Оповещение поисковиков о изменении карты сайта
		if( (bool)get_option('xmlsitemap_se_ping') ) {
			$sitemapUrl = urlencode( get_option('home') . '/' . $this->options['sitemap_file_name'] );
			$searchEngines = [
				'http://google.com/webmasters/sitemaps/ping?sitemap=' . $sitemapUrl,
				'http://www.bing.com/webmaster/ping.aspx?siteMap=' . $sitemapUrl,
			];
			foreach( $searchEngines as $searchEngine ) {
				file_get_contents( $searchEngine, 80, null, null, 0 );
			}
		}
	}

} // end SeoSitemap