<?php
/**
 * Рейтинг 5 звезд.
 *
 * 1. Рейтинг единичных записей.
 * 2. Рейтинг таксономий.
 *
 * Записывает в мета поля оценку статьи и количество голосов.
 * Сохраняет уникальные данные пользователя (IP,браузер) в базе данных ассоциированные с идентификатором записи или таксономии.
 * Создает нужные таблицы автоматически.
 *
 *  
 * @author veselka.ua
 * @version 0.12
 *
 * @package veselka_ua/themes
 */

class RatingStarSet {

	/**
	 * Массив параметров.
	 * @type array
	 */
	private $options;
	/**
	 * Массив POST после обработки.
	 * @type array
	 */
	private $post_data;
	/**
	 * Результат работы.
	 * @type string
	 */
	private $result_set;
	/**
	 * Ответ для JS.
	 * @type string
	 */
	private $answer;
	/**
	 * Лог работы для выброса в JS.
	 * Отладочный механизм, содержимое будет показано в модальном окне, при удачноом сохранениии рейтинга в базе данных.
	 * $this->logs .= "\n" . 'input_text_assembly: ' . $argument;
	 *
	 * @type string
	 */
	private $logs;


	public function __construct() {
		// Log
		$this->logs = '`\(>ᴗ<)/`    ' . 'Construct object.' . "\n";
	}


//////////////////// Публичные методы ////////////////////


	/**
	 * Обертка для получения результата.
	 *
	 * @param void
	 * @return string
	 */
	final public function get_result() {
		// Инициализация
		$this->load();
		// Результат
		return [ 'result' => $this->result_set, 'answer' => $this->answer, 'log' => $this->logs ];
	}


	/**
	 * Обертка для получения лога работы объекта.
	 *
	 * @param void
	 * @return string
	 */
	final public function get_logs() {
		// Инициализация
		$this->load();
		// Лог
		return $this->logs;
	}


//////////////////// Приватные методы ////////////////////


	/**
	 * Инициализация.
	 *
	 * @param void
	 * @return void
	 */
	private function load() {

		// Установка маркера
		$this->result_set = true;
		$this->answer = '┌∩┐(◣_◢)┌∩┐ ' . 'A rating error was detected!';
		// Адрес клиента
		$this->options['user_ip'] = md5($_SERVER['REMOTE_ADDR']);
		$this->options['user_agent'] = md5($_SERVER['HTTP_USER_AGENT']);
		// Параметры по умолчанию
		$this->options['post_votes_table_name'] = 'star_rating_post';
		$this->options['tax_votes_table_name'] = 'star_rating_tax';

		// Обработка POST
		$this->post_data_handling();
		// Проверка наличия адреса в базе
		$this->check_incoming_data();

		// Првоерка наличия соответствия ip - id
		if( $this->result_set ) {
			$this->check_unique();
		}

		// Запись рейтинга в базу
		if( $this->result_set ) {
			$this->record_new_rating();
		}
	
		// Проверка текущего рейтинга (получение данных рейтинга)
		if( $this->result_set ) {
			$this->check_new_rating();
		}

		// Возникла ошибка? Меняем ответ от сервера
		if( $this->result_set ) {
			// Ответ сервера
			$this->answer = '`\(>ᴗ<)/`    ' . 'Rating set!';
			$this->answer .= ' New rating: ' . $this->options['total'] . ' (total) / ' . $this->options['votes'] . ' (votes)';
		}
	}


	/**
	 * Обработка POST данных и запись в локальный массив.
	 *
	 * @param void
	 * @return void
	 */
	private function post_data_handling() {
		$this->post_data = [];
		if( isset($_POST) && !empty($_POST) ) {
			// Резервирование данных
			foreach( $_POST as $key => $value ) {
				$this->post_data[$key] = htmlspecialchars($value, ENT_NOQUOTES, 'UTF-8');
			}
			unset($_POST);
		}
		// Log
		$this->logs .= '`\(>ᴗ<)/`    ' . 'POST data processed!' . "\n";
	}


	/**
	 * Проверка наличия адреса в базе.
	 *
	 * @param void
	 * @return bool
	 */
	private function check_incoming_data() {

		// Флаг исключений
		$ex = false;

		$this->post_data['id'] = (int)$this->post_data['id'];
		$this->post_data['rate'] = (int)$this->post_data['rate'];

		// Проверка режима работы
		if( !in_array($this->post_data['mode'], ['post','tax']) ) {
			$this->logs .= '┌∩┐(◣_◢)┌∩┐ ' . 'No or wrong mode record!' . "\n";
			$ex = true;
		}

		// Проверка наличия нужной записи
		if( $this->post_data['mode'] == 'post' && !get_post($this->post_data['id']) ) {
			$this->logs .= '┌∩┐(◣_◢)┌∩┐ ' . 'Wrong post ID: ' . $this->post_data['id'] . "\n";
			$ex = true;
		}

		// Проверка наличия нужной таксономии
		if( $this->post_data['mode'] == 'tax' && !get_term($this->post_data['id']) ) {
			$this->logs .= '┌∩┐(◣_◢)┌∩┐ ' . 'Wrong taxonomy ID: ' . $this->post_data['id'] . "\n";
			$ex = true;
		}

		// Проверка рейтинга
		if( $this->post_data['rate'] > 5 ) {
			$this->logs .= '┌∩┐(◣_◢)┌∩┐ ' . 'Wrong rating value: ' . $this->post_data['rate'] . "\n";
			$ex = true;
		}

		// Ошибка?
		if( $ex ) {
			$this->result_set = false;
			return false;
		}

		// Log
		$this->logs .= '`\(>ᴗ<)/`    ' . 'Incoming data verified!' . "\n";
	}


	/**
	 * Проверка наличия адреса в базе.
	 *
	 * @param void
	 * @return bool
	 */
	private function check_unique() {
		global $wpdb;
		$table_name = $wpdb->prefix . $this->options[$this->post_data['mode'].'_votes_table_name'];
		$rating = $wpdb->get_row('SELECT `id` FROM `'. $table_name .'` WHERE id_'.$this->post_data['mode'].' = ' . $this->post_data['id'] . ' AND ip = "' . $this->options['user_ip'] . '" AND agent = "' . $this->options['user_agent'] . '"');
		if( $rating != NULL ) {
			$this->result_set = false;
			$this->answer = '((╬◣﹏◢))   ' . 'This User is already associated with the current record: ' . $_SERVER['REMOTE_ADDR'] . ' - ' . $_SERVER['HTTP_USER_AGENT'] . "\n";
			$this->logs .= $this->answer;
			return false;
		} else {
			$this->logs .= '`\(>ᴗ<)/`    ' . 'This user did not evaluate this record: ' . $_SERVER['REMOTE_ADDR'] . ' - ' . $_SERVER['HTTP_USER_AGENT'] . "\n";
		}
	}


	/**
	 * Запись рейтинга.
	 *
	 * @param void
	 * @return void
	 */
	private function record_new_rating() {

		// Получение рейтинга
		if( !$this->read_rating_from_meta() ) {
			return false;
		}

		// Добавление записи о рейтинге
		global $wpdb;
		$table_name = $wpdb->prefix . $this->options[$this->post_data['mode'].'_votes_table_name'];
		$data = [
			'id_'.$this->post_data['mode'] => $this->post_data['id'],
			'ip' => $this->options['user_ip'],
			'agent' => $this->options['user_agent']
		];
		if( $wpdb->insert( $table_name, $data ) ) {
			$this->logs .= '`\(>ᴗ<)/`    ' . 'Recorded user unique data in database: ' . $this->options['total'] . ' (total) / ' . $this->options['votes'] . ' (votes).' . "\n";
		} else {
			$this->logs .= '┌∩┐(◣_◢)┌∩┐ ' . 'Database error! User unique data not recorded!' . "\n";
		}

		// Вычисление нового среднего значения
		$summ_votes = $this->options['total'] * $this->options['votes'];
		$summ_votes += $this->post_data['rate'];
		$this->options['votes']++;
		$this->options['total'] = round( $summ_votes/$this->options['votes'], 5 );

		// Запись нового значения в метаполя
		$this->write_rating_to_meta();

	}


	/**
	 * Проверка текущего рейтинга.
	 * 
	 * Получение из метаданных для мгновенного контороля результата.
	 *
	 * @param void
	 * @return void
	 */
	private function check_new_rating() {

		// Получение рейтинга
		if( $this->read_rating_from_meta() ) {
			// Результат работы для JS
			$this->result_set = [];
			$this->result_set['total'] = round( $this->options['total'], 1 );
			$this->result_set['votes'] = $this->options['votes'];
		} else {
			$this->logs .= '┌∩┐(◣_◢)┌∩┐ ' . 'Rating not set: ' . $this->options['total'] . ' (total) / ' . $this->options['votes'] . ' (votes).' . "\n";
		}
	}


	/**
	 * Получение значений рейтинга из метаданных.
	 *
	 * @param void
	 * @return bool
	 */
	private function read_rating_from_meta() {
		// Получение текущего рейтинга
		if( $this->post_data['mode'] == 'post' ) {
			$this->options['total'] = get_post_meta( (int)$this->post_data['id'], 'rating_total', true );
			$this->options['votes'] = get_post_meta( (int)$this->post_data['id'], 'rating_votes', true );
		} elseif( $this->post_data['mode'] == 'tax' ) {
			$this->options['total'] = get_term_meta( (int)$this->post_data['id'], 'cat_rating_total', true );
			$this->options['votes'] = get_term_meta( (int)$this->post_data['id'], 'cat_rating_votes', true );
		} else {
			$this->logs .= '┌∩┐(◣_◢)┌∩┐ ' . 'No mode record in read_rating_from_meta() method!' . "\n";
			$this->result_set = false;
		}

		// Нет рейтинга, обработка исключения
		if( !(bool)$this->options['total'] || !(bool)$this->options['votes'] ) {
			$this->logs .= '┌∩┐(◣_◢)┌∩┐ ' . 'No rating data with '.$this->post_data['mode'].' ID: ' . $this->post_data['id'] . "\n";
			$this->result_set = false;
			return false;
		}

		// Все ок
		$this->logs .= '`\(>ᴗ<)/`    ' . 'Rating successfully read: ' . $this->options['total'] . ' (total) / ' . $this->options['votes'] . ' (votes).' . "\n";
		return true;
	}


	/**
	 * Запись значений рейтинга в метаданные.
	 *
	 * @param void
	 * @return bool
	 */
	private function write_rating_to_meta() {
		// Запись нового значения
		if( $this->post_data['mode'] == 'post' ) {
			update_post_meta( (int)$this->post_data['id'], 'rating_total', $this->options['total'] );
			update_post_meta( (int)$this->post_data['id'], 'rating_votes', $this->options['votes'] );
		} elseif( $this->post_data['mode'] == 'tax' ) {
			update_term_meta( (int)$this->post_data['id'], 'cat_rating_total', $this->options['total'] );
			update_term_meta( (int)$this->post_data['id'], 'cat_rating_votes', $this->options['votes'] );
		} else {
			$this->logs .= '┌∩┐(◣_◢)┌∩┐ ' . 'No mode record in write_rating_to_meta() method!' . "\n";
			$this->result_set = false;
			return false;
		}

		// Все ок
		$this->logs .= '`\(>ᴗ<)/`    ' . 'Recorded in the meta field: ' . $this->options['total'] . ' (total) / ' . $this->options['votes'] . ' (votes).' . "\n";
		return true;
	}

} // end RatingStarSet
?>