<?php
/**
 * Вспомогательные кнопки.
 * 
 * Формирует разметку статических кнопок справа.
 * Загружает необходимые библиотеки.
 * 
 * @param object $lib_construct
 * @param object $body_applications
 * @param object $open_graph
 *
 * @author veselka.ua
 * @version 0.1
 */

class HelpButtons {

	use FileHandling, TemplatesLoad;

	/**
	 * Объект обработки библиотек.
	 * @type object
	 */
	private $lib_kit;
	/**
	 * Объект загрузки скриптов в начале <body>.
	 * @type object
	 */
	private $app_kit;
	/**
	 * Массив параметров.
	 * @type array
	 */
	private $options;
	/**
	 * Результат работы.
	 * @type string
	 */
	private $result_set;


	public function __construct() {
		// Инициализация
		$this->options['init'] = false;

		/**
		 * Хук для получение объекта с общими методами.
		 */
		add_action( 'wp_lib_construct_exchange', [$this, 'get_wp_lib_construct'], 21, 1 );

		/**
		 * Хук для получение объекта с общими методами.
		 */
		add_action( 'wp_applications_exchange', [$this, 'get_wp_applications'], 10, 1 );

		// В крючке вызывается метод обертка
		add_action( 'help_buttons', [$this, 'get_result_set'], 20, 0 );

	}


//////////////////// Публичные методы ////////////////////


	/**
	 * функция хука для получения общего объекта по ссылке.
	 *
	 * Сортировка.
	 *
	 * @param object $lib_construct
	 * @return void
	 */
	final public function get_wp_lib_construct( $lib_construct ){
		// Объект с общими методами в локальное свойство
		$this->lib_kit = $lib_construct;
	}


	/**
	 * функция хука для получения общего объекта по ссылке.
	 *
	 * Сортировка.
	 *
	 * @param object $applications
	 * @return void
	 */
	final public function get_wp_applications( $applications ){
		// Объект с общими методами в локальное свойство
		$this->app_kit = $applications;
	}


	/**
	 * Оберткка для получения результата работы
	 *
	 * @param void
	 * @return string
	 */
	public function get_result_set() {

		// Инициализация
		$this->load();
		// Вспомогательные кнопки - формирование разметки
		$this->set_help_buttons_html();
		// Вывод результата
		echo $this->result_set;
	}


//////////////////// Приватные методы ////////////////////


	/**
	 * Инициализация.
	 *
	 * @param void
	 * @return void
	 */
	private function load() {

		// Уже загружалось
		if( $this->options['init'] ) {
			return true;
		}
		// Параметры файлов шаблонов по умолчанию
		$this->options['lib'] = false;
		$this->options['files']['main'] = 'main.php';
		$this->options['files']['scrollup'] = 'scrollup.php';
		$this->options['files']['scrolldown'] = 'scrolldown.php';
		$this->options['files']['help_buttons_item'] = 'help_buttons_item.php';
		$this->options['files']['search_form'] = 'search_form.php';
		$this->options['files']['social_buttons'] = 'social_buttons.php';
		$this->options['files']['social_buttons_item'] = 'social_buttons_item.php';
		// Загрузка библиотек
		if( !$this->options['lib'] ) {
			$this->lib_kit->set_new_lib('help_buttons');
			$this->options['lib'] = true;
		}
		// Массив маркеров приложений лдя социальных кнопок
		$this->options['social'] = [
			'fb'	=>	[
				'ico'	=>	'fa-facebook',
				'name'	=>	'facebook',
				'app'	=>	false,
			],
			'vk'	=>	[
				'ico'	=>	'fa-vk',
				'name'	=>	'vkontakte',
				'app'	=>	false,
			],
			'pi'	=>	[
				'ico'	=>	'fa-pinterest-p',
				'name'	=>	'pinterest',
				'title'	=>	__('Share on Pinterest','spiral'),
			],
			'gp'	=>	[
				'ico'	=>	'fa-google-plus',
				'name'	=>	'gplus',
				'app'	=>	false,
			],
		];
		// Загрузка приложений для соцкнопок, проверка наличия key Id
		foreach( $this->options['social'] as $id => $data ) {
			// Нужно загруить приложение
			if( isset($data['app']) && $data['app'] ) {
				if( !(bool)get_option('api_' . $data['name']) ) {
					unset($options['social'][$id]);
					continue;
				}
				// Загрузка кода приложения
				$this->app_kit->set_new_app($data['name']);
			}
		}

		// Инициализация завершена
		$this->options['init'] = true;
	}

	/**
	 * Установка переменных.
	 *
	 * @param void
	 * @return void
	 */
	private function set_variables() {
		/**
		 * Стандартный путь к параметрам.
		 */
		require_once TEMPLATEPATH . '/conf/help_buttons.php';

		// Массив вспомогательных кнопок
		$this->options['help_btns'] = $help_btns;

		// Переводы социальных кнопок
		if( isset($this->options['social']['fb']) ) {
			$this->options['social']['fb']['title'] = __('Share on Facebook','spiral');
		}
		if( isset($this->options['social']['vk']) ) {
			$this->options['social']['vk']['title'] = __('Share on Vkontakte','spiral');
		}
		if( isset($this->options['social']['pi']) ) {
			$this->options['social']['pi']['title'] = __('Share on Pinterest','spiral');
		}
		if( isset($this->options['social']['gp']) ) {
			$this->options['social']['gp']['title'] = __('Share on Google+','spiral');
		}
	}


	/**
	 * Формирование разметки кнопок.
	 *
	 * @param void
	 * @return void
	 */
	private function set_help_buttons_html() {
		// Установка переменных
		$this->set_variables();
		// Вспомогательные кнопки
		$replacement['common_buttons'] = $this->set_common_buttons_html();
		// Социальные кнопки
		$replacement['social_buttons'] = $this->set_social_buttons_html();
		// Замена маркеров значениями
		$this->result_set = $this->template_replacement( $replacement, 'main', 'help_buttons' );
	}


	/**
	 * Формирование разметки вспомогательных кнопок.
	 *
	 * @param void
	 * @return string
	 */
	private function set_common_buttons_html() {

		// Запись кнопок
		$result_set = '';
		foreach( $this->options['help_btns'] as $key => $btn ) {
			// Включена ли кнопка в админпенели
			if( !(bool)get_option('btns_' . $key) ) {
				continue;
			}
			// Массив замен
			$replacement['btn_id'] = $btn['id'];
			$replacement['btn_ico'] = $btn['ico'];
			$replacement['btn_title'] = $btn['title'];
			$replacement['btn_class'] = $btn['class']!=''?$btn['class'] . ' ':'';
			$replacement['btn_toggle'] = $btn['data']['toggle'];
			$replacement['btn_target'] = $btn['data']['target'];
			$replacement['btn_content'] = $btn['data']['content'];
			// Если указана функция построения дата атрибута кнопки
			if( $replacement['btn_content'] != '' && method_exists( $this, $replacement['btn_content'] ) ) {
				$replacement['btn_content'] = $this->{$replacement['btn_content']}($btn['title']);
			} else {
				$replacement['btn_content'] = '';
			}
			// Скрытая кнопка
			$replacement['style'] = '';
			if( $btn['hidden'] ) {
				$replacement['style'] = ' style="display:none;"';
			}
			// Шаблон вспомогательных кнопок
			$template = 'help_buttons_item';
			if( in_array( $key, ['scrollup','scrolldown'] ) ) {
				$template = $key;
			}
			$result_set .= $this->template_replacement( $replacement, $template, 'help_buttons' );
		}
		return $result_set;
	}


	/**
	 * Формирование разметки формы поиска.
	 *
	 * @param void
	 * @return string
	 */
	private function set_search_button_html($title) {
		
		// Шаблон формы поиска
		$search = $this->template_replacement( ['url'=>esc_url(home_url( '/' )),'placeholder'=>__('Search','spiral').'...'], 'search_form', 'help_buttons' );
		$search = "\t\t\t" . 'data-content="' . $search . '"';
		$search .= PHP_EOL . "\t\t\t" . 'data-original-title="' . $title . '"';
		return $search;
	}



	/**
	 * Формирование разметки социальных кнопок.
	 *
	 * @param void
	 * @return string
	 */
	private function set_social_buttons_html() {
		// Социальные кнопки включены
		if( !(bool)get_option('btns_social') ) {
			return '';
		}

		// Запись кнопок
		$result_set = '';
		foreach( $this->options['social'] as $id => $data ) {
			// Нужно загруить приложение
			if( $data['app'] ) {
				if( !(bool)get_option('api_' . $data['name']) ) {
					continue;
				}
			}
			// Массив замен
			$replacement['btn_id'] = 'share-' . $id;
			$replacement['btn_ico'] = $data['ico'];
			$replacement['btn_title'] = $data['title'];
			// Шаблон соцкнопок
			$result_set .= $this->template_replacement( $replacement, 'social_buttons_item', 'help_buttons' );
		}
		// Нет соцкнопок
		if( $result_set == '' ) {
			return '';
		}
		// Массив замен
		$replacement['social_buttons'] = $result_set;
		$replacement['hide_title'] = __('Hide social buttons','spiral'); 
		$replacement['share_title'] = __('Share on social networks','spiral');
		// Шаблон кнопок поиска
		$result_set = $this->template_replacement( $replacement, 'social_buttons', 'help_buttons' );
		// Результат
		return $result_set;
	}




} // end HelpButtons
?>