<?php
/**
 * Импорт параметров единичных записей.
 *
 * Импортирует параметры из Google Spreadsheets в базу данных WordPress.
 *
 * @param obj $gdrive
 *
 * @author veselka.ua
 * @version 0.1
 *
 * @package veselka_ua/themes
 */

class GdriveImportSingular extends GdriveImportCommon {


	public function __construct( &$gdrive ) {
		// Объект инструментов Google Spreadsheets
		if( !$gdrive && !is_object($gdrive) ) {
			throw new \Exception('No Google tools obj');
		}
 
		// Запись объекта в свойство
		$this->gdrive = $gdrive;
		// Флаг
		$this->registry['single'] = true;
	}


/////////////////// Публичные методы ////////////////////


	/**
	 * Обертка для получения ключа.
	 *
	 * @param string $destination
	 * @return void
	 */
	final public function import_data( $destination='' ) {
		// Имя сущности
		$this->registry['destination'] = $destination;
		// Развертывание (Метод в родительском объекте)
		$this->load();
		// Перебор строк
		$this->parse_rows();
		// Запись обновленной таблицы в файл
		$this->gdrive->write_file_data();
	}


//////////////////// Приватные методы ////////////////////


	/**
	 * Перебор строк файла.
	 *
	 * @param void
	 * @return void
	 */
	private function parse_rows() {
		// Перебор значений
		while( list($row, $data) = $this->gdrive->get_file_data() ) {

			// Обнуление переменных
			$this->registry['up_to_date'] = false;
			$this->registry['post'] = true;
			$this->registry['data'] = [];
			// Текущая строка
			foreach($this->index as $associative_key => $numeric_key ) {
				$this->registry['data'][$associative_key] = $data[$numeric_key];
			}
			$this->registry['data']['import_status'] = '';
			// Запуск методов в зависимости от состава файла загрузки
			foreach( $this->registry['methods'] as $name => $arg ) {
				// Формирование имени метода
				$method_name = $name . '_field_construct';
				// Вызов соответствующего метода для сбора данных
				if( method_exists( $this, $method_name ) ) {
					if( is_array($arg) && !empty($arg) ) {
						$this->{$method_name}($arg);
					} else {
						$this->{$method_name}();
					}
				}
				if( !$this->registry['post'] ) {
					next;
				}
			}

			// Запись результата
			if( $this->registry['post'] ) {
				$this->create_record();
			}
			// Запись статуса
			$this->write_status();
			// Запись обновленных значений 
			foreach($this->index as $associative_key => $numeric_key ) {
				// Перевод в целочисленный формат
				if( in_array($associative_key, ['wp_id','tax_id','wp_order']) ) {
					$data[$numeric_key] = (int)$this->registry['data'][$associative_key];
				} else {
					$data[$numeric_key] = $this->registry['data'][$associative_key];
				}
			}
			// Запись в файл обновленной строки
			$this->gdrive->set_file_data($row, $data);
		}
		//var_dump($this->registry['data']);
	}


	/**
	 * Проверка наличия записи по дентификатору.
	 *
	 * @param void
	 * @return void
	 */
	private function id_field_construct() {
		if( $this->registry['data']['wp_id'] !== '' && get_post_status($this->registry['data']['wp_id']) !== false ) {
			$this->registry['wp_post']['ID'] = $this->registry['data']['wp_id'];
		} else {
			$this->registry['data']['wp_id'] = '';
		}
	}


	/**
	 * Проверка наличия записи о ярлыку.
	 *
	 * @param array $data
	 * @return void
	 */
	private function slug_field_construct() {
		// Проверка идентификатора
		$this->id_field_construct();
		// Проверка ярлыка
		if( empty($this->registry['data']['wp_id']) ) {
			if( empty($this->registry['data']['wp_slug']) ) {
				// Параметры не переданы, новая запись
				$this->load_staus('status_new_slug');
				// Если существует метод описывающий генерацию ярлыка
				$method_name = $this->registry['destination'] . '_slug_construct';
				// Вызов соответствующего метода для сбора данных
				if( method_exists( $this, $method_name ) ) {
					$this->{$method_name}();
				} else {
					// Генератор ярлыка по умолчанию (транслитерация заголовка или h1)
					if( !empty($this->registry['data']['h1']) ) {
						$this->registry['data']['wp_slug'] = apply_filters( 'url_transliteration', $this->registry['data']['h1'] );
					} elseif( !empty($this->registry['data']['title']) ) {
						$this->registry['data']['wp_slug'] = apply_filters( 'url_transliteration', $this->registry['data']['title'] );
					}
				}
				if( empty($this->registry['data']['wp_slug']) ) {
					// Slug не сгенерирован
					$this->load_staus('error');
					$this->load_staus('warn_no_wpslug_gen');
					// Уничтожение объекта - пеерход к следующей строке
					$this->registry['post'] = false;
				}
			} else {
				// Восстановление id из slug
				$post = get_page_by_path(strtolower((string)$this->registry['data']['wp_slug']), OBJECT, $this->registry['destination']);
				if( is_object($post) ) {
					$this->registry['post'] = $post;
					$this->registry['data']['wp_id'] = $this->registry['post']->ID;
					// Запись id в массив добавления записи
					$this->registry['wp_post']['ID'] = $this->registry['post']->ID;
					// Запись ярлыка в массив добавления записи
					$this->registry['wp_post']['post_name'] = $this->registry['post']->post_name;
				} else {
					// Нет такого ярлыка, новая запись
					$this->load_staus('status_new');
					// Ярлык
					$this->registry['wp_post']['post_name'] = $this->registry['data']['wp_slug'];
				}
			}
		} else {
			$this->registry['post'] = get_post($this->registry['data']['wp_id']);
			if( empty($this->registry['data']['wp_slug']) ) {
				// Восстановление slug из id
				$this->registry['data']['wp_slug'] = $this->registry['post']->post_name;
			} else {
				// Оба параметра переданы, сравнение
				// Восстановление id из slug
				$post = get_page_by_path($this->registry['data']['wp_slug'], OBJECT, $this->registry['destination']);
				if( $post->ID != $this->registry['data']['wp_id'] ) {
					// Slug ротиворечит ID
					$this->load_staus('error');
					$this->load_staus('warn_bad_wpslug');
					// Уничтожение объекта - пеерход к следующей строке
					$this->registry['post'] = false;
				}
			}
			// Запись id в массив добавления записи
			$this->registry['wp_post']['ID'] = $this->registry['post']->ID;
			// Запись ярлыка в массив добавления записи
			$this->registry['wp_post']['post_name'] = $this->registry['post']->post_name;
		}
	}


	/**
	 * Проверка наличия родительской таксономии.
	 *
	 * @param array $arg
	 * @return void
	 */
	private function tax_field_construct() {
		// Страница
		if( $this->registry['destination'] == 'page' ) {
			return NULL;
		}
		// Обработка полей таксономии
		if( empty($this->registry['data']['tax_id']) ) {
			// Параметры не переданы, нет таксономии
			if( !isset($this->registry['data']['tax_slug']) || empty($this->registry['data']['tax_slug']) ) {
				// Попытка восстановления идентификатора таксономии из объекта записи.
				if( !is_object($this->registry['post']) ) {
					// Параметры не переданы, нет таксономии
					$this->load_staus('error');
					$this->load_staus('warn_no_tax');
					// Уничтожение объекта - пеерход к следующей строке
					$this->registry['post'] = false;
					return NULL;
				}
				// Объект таксономии по идентификатору поста
				$tax_p = get_the_terms( $this->registry['post']->ID, $this->registry['tax'] );
				if( is_wp_error($tax_p) || !$tax_p ) {
					// Ошибка получения таксономии
					$this->load_staus('error');
					$this->load_staus('warn_no_tax_res');
					// Уничтожение объекта - пеерход к следующей строке
					$this->registry['post'] = false;
					return NULL;
				}
				// Востановление параметров
				foreach( $tax_p as $tax ) {
					$this->registry['data']['tax_id'] = (int)$tax->term_id;
					$this->registry['data']['tax_slug'] = $tax->slug;
					//$this->registry['wp_post']['tax_input'][$this->registry['tax']][] = $tax->slug;
				}
			} else {
				// Восстановление id из slug
				$tax = get_term_by('slug', $this->registry['data']['tax_slug'], $this->registry['tax']);
				if( !$tax ) {
					// Ошибочный ярлык таксономии
					$this->load_staus('error');
					$this->load_staus('warn_bad_tax_slug');
					// Уничтожение объекта - пеерход к следующей строке
					$this->registry['post'] = false;
					return NULL;
				}
				$this->registry['data']['tax_id'] = (int)$tax->term_id;
				// Запись id таксономии в массив добавления записи
				//$this->registry['wp_post']['tax_input'][$this->registry['tax']][] = $tax->slug;
			}
		} elseif( isset($this->registry['data']['tax_slug']) ) {
			// Восстановление slug из id
			$tax_i = get_term($this->registry['data']['tax_id']);
			// Проверка полученной таксономии
			if( is_wp_error($tax_i) ) {
				// Ошибочный идентификатор таксономии
				$this->load_staus('error');
				$this->load_staus('warn_bad_tax_id');
				// Уничтожение объекта - пеерход к следующей строке
				$this->registry['post'] = false;
				return NULL;
			}
			// Пустое поле таксономии?
			if( empty($this->registry['data']['tax_slug']) ) {
				// Восстановление slug из id
				$this->registry['data']['tax_id'] = (int)$tax_i->term_id;
				// Запись id таксономии в массив добавления записи
				//$this->registry['wp_post']['tax_input'][$this->registry['tax']][] = $tax_i->slug;
			} else {
				// Оба параметра переданы, сравнение
				// Восстановление id из slug
				$tax_s = get_term_by('slug', $this->registry['data']['tax_slug'], $this->registry['tax']);
				if( !$tax_s ) {
					// Ошибочный ярлык таксономии
					$this->load_staus('error');
					$this->load_staus('warn_bad_tax_slug');
					// Уничтожение объекта - пеерход к следующей строке
					$this->registry['post'] = false;
					return NULL;
				}
				if( $tax_i->term_id != $tax_s->term_id ) {
					// Slug ротиворечит ID
					$this->load_staus('error');
					$this->load_staus('warn_bad_tax');
					// Уничтожение объекта - пеерход к следующей строке
					$this->registry['post'] = false;
					return NULL;
				}
				// Запись id таксономии в массив добавления записи
				//$this->registry['wp_post']['tax_input'][$this->registry['tax']][] = $tax_i->slug;
			}
		}
	}


	/**
	 * Проверка актуальности записи.
	 * 
	 *
	 * @param void
	 * @return void
	 */
	private function date_field_construct() {
		// Нет даты модификации
		if( $this->registry['data']['modification_time'] == '' ) {
			return true;
		}
		// Дата из файла
		$file_timestamp = new DateTime($this->registry['data']['modification_time']);
		$file_date = $file_timestamp->format('Y-m-j H:i:s');
		$file_unix = strtotime($file_date);

		// Нет записи - пишем дату из базы и досвидос
		if( !is_object($this->registry['post']) ) {
			// Новый пост дату создания из файла
			$this->registry['wp_post']['post_date'] = $file_date;
			return true;
		}
		// Дата из базы
		$db_timestamp = new DateTime($this->registry['post']->post_date);
		$db_date = $db_timestamp->format('Y-m-j H:i:s');
		$db_unix = strtotime($db_date);
		// Сравнение даты из базы
		if( $file_unix > $db_unix ) {
			// Дата в файле старше, записываем
			$this->registry['wp_post']['post_date'] = $file_date;
		} else if( $file_unix == $db_unix ) {
			// Пост изменен позднее, не пишем в базу
			$this->load_staus('status_actual');
			$this->registry['up_to_date'] = true;
		} else {
			// Пост изменен позднее, не пишем в базу
			$this->load_staus('warning');
			$this->load_staus('warn_old_date');
			// Уничтожение объекта - пеерход к следующей строке
			$this->registry['post'] = false;
		}
	}


	/**
	 * Сортировка записи.
	 * 
	 *
	 * @param void
	 * @return void
	 */
	private function sort_field_construct() {
		if( $this->registry['data']['wp_order'] != '' ) {
			$this->registry['wp_post']['menu_order'] = (int)$this->registry['data']['wp_order'];
		} else {
			// Восстановление значения из базы
			if( is_object($this->registry['post']) && $this->registry['post']->menu_order != 0 ) {
				$this->registry['data']['wp_order'] = $this->registry['post']->menu_order;
			}
		}
	}


	/**
	 * Статус записи.
	 * 
	 *
	 * @param void
	 * @return void
	 */
	private function status_field_construct() {
		if( isset($this->registry['data']['wp_status']) &&
			$this->registry['data']['wp_status'] != '' &&
			in_array( $this->registry['data']['wp_status'], ['draft', 'publish', 'pending', 'future', 'private'] )
		) {
			$this->registry['wp_post']['post_status'] = $this->registry['data']['wp_status'];
		} else {
			// Восстановление значения из базы
			if( is_object($this->registry['post']) ) {
				$this->registry['wp_post']['post_status'] = $this->registry['post']->post_status;
			} else {
				// Новая запись, в файле ничего не указано - публикуем
				$this->registry['wp_post']['post_status'] = 'publish';
			}
		}
	}


	/**
	 * Запись заголовка.
	 * 
	 *
	 * @param void
	 * @return void
	 */
	private function title_field_construct() {
		if( $this->registry['data']['title'] != '' ) {
			$this->registry['wp_post']['post_title'] = $this->registry['data']['title'];
		}
	}


	/**
	 * Запись краткого описания.
	 * 
	 *
	 * @param void
	 * @return void
	 */
	private function excerpt_field_construct() {
		if( $this->registry['data']['excerpt'] != '' ) {
			$this->registry['wp_post']['post_excerpt'] = $this->registry['data']['excerpt'];
		}
	}


	/**
	 * Запись полного описания.
	 * 
	 *
	 * @param void
	 * @return void
	 */
	private function content_field_construct() {
		if( $this->registry['data']['content'] != '' ) {
			$this->registry['wp_post']['post_content'] = $this->registry['data']['content'];
		}
	}


	/**
	 * Запись SEO параметров.
	 * 
	 *
	 * @param array $args
	 * @return void
	 */
	private function seo_field_construct($args) {
		foreach($args as $key) {
			$this->registry['meta_input'][$key] = $this->registry['data']['seo_'.$key];
		}
	}


	/**
	 * Запись мета параметров.
	 * 
	 *
	 * @param array $args
	 * @return void
	 */
	private function meta_field_construct($args) {
		foreach($args as $key) {
			$this->registry['meta_input']['meta_'.$key] = $this->registry['data']['meta_'.$key];
		}
	}


	/**
	 * Создание записи.
	 * 
	 *
	 * @param void
	 * @return void
	 */
	private function create_record() {
		// Нужно ли обновление
		if( $this->registry['up_to_date'] ) {
			return NULL;
		}
		// Параметры для создания/обновления записи
		$this->registry['wp_post']['post_type'] = $this->registry['destination'];
		$this->registry['wp_post']['post_author'] = 1;
		// Запись данных
		$post_id = wp_insert_post($this->registry['wp_post']);
		if( $post_id === 0 || !$post_id ) {
			// Пост изменен позднее, не пишем в базу
			$this->load_staus('error');
			$this->load_staus('warn_write_err');
			// Уничтожение объекта - пеерход к следующей строке
			$this->registry['post'] = false;
			return NULL;
		}

		update_post_meta($post_ID, '_wp_page_template',  $page_template);
		// Статус записи
		if( $this->registry['post'] === true ) {
			$this->load_staus('status_create');
			// Дописываем в таблицу идентификатор вновь созданной записи
			$this->registry['data']['wp_id'] = $post_id;
			// Запись таксономии
			wp_set_post_terms( $post_id, [$this->registry['data']['tax_id']], $this->registry['tax'] );
		} else {
			// Обновление метаданных
			foreach( $this->registry['meta_input'] as $key => $value ) {
				if( $value != '' ) {
					update_metadata('post', $post_id, $key, $value);
				} else {
					delete_metadata('post', $post_id, $key);
				}
			}
			// Статус
			$this->load_staus('status_update');
		}
		//var_dump($this->registry['data']);
	}


	/**
	 * Генерация ярлыка для квартир.
	 * 
	 *
	 * @param void
	 * @return void
	 */
	private function estate_slug_construct() {
		$slug = '';
		// Идентификатор ЖК
		if( isset($this->registry['data']['tax_slug']) && $this->registry['data']['tax_slug'] != '' ) {
			$slug .= $this->registry['data']['tax_slug'];
		}
		// Номер очереди
		if( isset($this->registry['data']['mata_section']) && $this->registry['data']['mata_section'] != '' ) {
			$slug .= '-s' . (string)$this->registry['data']['mata_section'];
		}
		// Номера этажей
		if( isset($this->registry['data']['mata_floor_availability']) && $this->registry['data']['mata_floor_availability'] != '' ) {
			$floors = explode('&',$this->registry['data']['mata_floor_availability']);
			foreach( $floors as $flor ) {
				$slug .= '-f' . $flor;
			}
		}
		// Номер квартиры
		if( isset($this->registry['data']['meta_room_num']) && $this->registry['data']['meta_room_num'] != '' ) {
			$slug .= '-r' . (string)$this->registry['data']['meta_room_num'];
		}
		// площадь квартиры
		if( isset($this->registry['data']['meta_s']) && $this->registry['data']['meta_s'] != '' ) {
			$squaere = explode(',',$this->registry['data']['meta_s']);
			$slug .= '-' . $squaere[0] . '_' . $squaere[1];
		}
		// результат
		$this->registry['data']['wp_slug'] = $slug;
	}






} //end GdriveImportSingular
?>