<?php
/**
 * Лендинг.
 *
 * Таксономия элементов лендинга по умолчанию.
 * Добавляет таксономию по умолчаюнию для лендинга (по типу "Без категории" для стандартных записей).
 * Присваивает таксономию по умолчаюнию при сохранении, если ничего не выбрано.
 *
 *
 * Для рабобы объекта необходимо добавить аргументы к функции register_taxonomy в файле plugins/veselka_landing/lib/registratio.class.php
 *
 * default_term			<string>	<обязательный>		- Имя таксономии по умолчанию
 * default_term_slug	<string>	<не обязательный>	- Ярлык таксономии по умолчанию
 * default_term_force	<bool>		<не обязательный>	- Установить таксономию независимо от того, выбрано ли значение по умолчанию, даже если пользователь отключает или отменяет выбор по умолчанию. По умолчанию true
 *
 *
 *
 * @author veselka.ua
 * @version 0.2
 *
 * @package veselka_ua/veselka_landing
 */

class LandingDefault {


	public function __construct() {
		/**
		 * Хук активации, создание таксономии по умолчанию.
		 */
		register_activation_hook( LANDING_PLUGIN_FILE, [$this, 'default_term_activation'] );

		/**
		 * Убираем ссылку удаления таксономии по умолчанию.
		 */
		add_action( 'admin_init', [$this, 'default_term_delete_link'] );

		/**
		 * Сохранение элемента лендинга.
		 */
		add_action( 'save_post', [$this, 'default_term_save_post'], 99999, 3 );
	}


//////////////////// Публичные методы ////////////////////


	/**
	 * Создает термин таксономии по умолчинию при активации плагина.
	 *
	 * @param void
	 * @return void
	 */
	final public function default_term_activation() {

		$taxonomy = get_taxonomy( 'landing' );

		// Проверка наличия термина по умолчанию
		if( $id = term_exists( $taxonomy->default_term, 'landing' ) ) {
			// Обновляем
			update_option( 'default_landing', $id['term_id'] );
			return true;
		}
		// Термина не существует - добавляем
		$term = wp_insert_term( $taxonomy->default_term, 'landing', [
			'slug' => $taxonomy->default_term_slug
		]);

		// Ошибка WP?
		if( is_wp_error($term) ) {
			return true;
		}
		// Сохраниение 
		update_option( 'default_landing', $term['term_id'] );

	}


	/**
	 * Убираем ссылку удаления термина по умолчанию в админпаели при выводе списка терминов.
	 *
	 * @param void
	 * @return void
	 */
	final public function default_term_delete_link() {

		if( !get_option('default_landing') ) {
			$this->default_term_activation();
		}

		global $wp_post_types;

		add_filter( 'landing_row_actions', create_function('$actions, $term', '
			if( $term->term_id != get_option("default_landing") )
			{
				return $actions;
			}
			unset($actions["delete"]);
			return $actions;
		'), 10, 2 );
	}


	/**
	 * Действия при сохранении элемента лендинга
	 *
	 * @param  int $post_id
	 * @param  WP_Post $post
	 * @param  boolean $update - Новый элемент лендинга или существующий
	 * @return void
	 */
	final public function default_term_save_post($post_id, $post, $update) {

		//global $wp_taxonomies, $wp_posttypes;

		// Запись не отправлена на сохранение
		if( empty($_POST) ) {
			return;
		}

		// Данные отправлены без tax_input
		if( !isset($_POST['tax_input']) ) {
			return;
		}

		// Данные не относятся к лендингу
		if( !isset($_POST['tax_input']['landing']) ) {
			return;
		}

		$values = $_POST['tax_input']['landing'];
		if( COUNT(array_filter($values)) > 0 ) {
			return;
		}

		// If there is no default term then don't to anything
		$default_term = get_option('default_landing');
		if( !get_option('default_landing') ) {
			return;
		}


		$term = get_term( $default_term, 'landing' );
		if( is_wp_error($term) ) {
			return;
		}
		$default_term = $term->name;

		// Else asign the default term if the post has no taxonomy terms.
		wp_set_post_terms( $post_id, $default_term, 'landing' );
	}


} // end LandingDefault
?>